import { forEach, map } from "lodash-es";

interface FileResult {
  file: File;
  dataUri: string;
}

const domUtil = {
  loadImage(url: string): Promise<HTMLImageElement> {
    return new Promise((resolve, reject) => {
      const image = new Image();
      image.crossOrigin = "anonymous";
      image.onload = () => resolve(image);
      image.onerror = reject;
      image.src = url;
    });
  },

  async combinePngs(urls: string[]): Promise<FileResult> {
    const loadImages = map(urls, (url) => this.loadImage(url));
    const images = await Promise.all(loadImages);
    const canvas = document.createElement("canvas");
    const { width, height } = images[0];
    canvas.width = width;
    canvas.height = height;
    const context = canvas.getContext("2d");
    if (!context) throw new Error("Context not found");
    forEach(images, (image) => {
      context.drawImage(image, 0, 0);
    });
    return this.canvasToPng(canvas);
  },

  canvasToPng(canvas: HTMLCanvasElement): Promise<FileResult> {
    return new Promise((resolve, reject) => {
      const dataUri = canvas.toDataURL("image/png");
      canvas.toBlob((blob) => {
        if (!blob) return reject();
        const file = new File([blob], "image.png", { type: "image/png" });
        resolve({ file, dataUri });
      });
    });
  },
};

export default domUtil;
